if __name__ == "__main__":
    import sys
    import os
    import pathlib

    ROOT_DIR = str(pathlib.Path(__file__).parent.parent.parent)
    sys.path.append(ROOT_DIR)
    os.chdir(ROOT_DIR)

import os
import hydra
import torch
from omegaconf import OmegaConf
import pathlib
from torch.utils.data import DataLoader
import copy
import random
import wandb
import tqdm
import numpy as np
import shutil
from diffusion_policy.workspace.base_workspace import BaseWorkspace
from diffusion_policy.policy.diffusion_realrobot_policy import DiffusionRealRobotPolicy
from diffusion_policy.dataset.base_dataset import BaseImageDataset
from diffusion_policy.env_runner.base_image_runner import BaseImageRunner
from diffusion_policy.common.checkpoint_util import TopKCheckpointManager
from diffusion_policy.common.json_logger import JsonLogger
from diffusion_policy.common.pytorch_util import dict_apply, optimizer_to
from diffusion_policy.model.diffusion.ema_model import EMAModel
from diffusion_policy.model.common.lr_scheduler import get_scheduler
from diffusion_policy.common.compute_all_loss import compute_all_traj_loss_realrobot

OmegaConf.register_new_resolver("eval", eval, replace=True)

class PbrlDiffusionRealRobotWorkspace(BaseWorkspace):
    include_keys = ['global_step', 'epoch']

    def __init__(self, cfg: OmegaConf, output_dir=None):
        super().__init__(cfg, output_dir=output_dir)

        # set seed
        seed = cfg.training.seed
        torch.manual_seed(seed)
        np.random.seed(seed)
        random.seed(seed)

        # configure model
        self.model: DiffusionRealRobotPolicy = hydra.utils.instantiate(cfg.policy)

        self.ema_model: DiffusionRealRobotPolicy = None
        if cfg.training.use_ema:
            self.ema_model = copy.deepcopy(self.model)

        # configure training state
        self.optimizer = hydra.utils.instantiate(
            cfg.optimizer, params=self.model.parameters())

        # configure training state
        self.global_step = 0
        self.epoch = 0

    def run(self):
        cfg = copy.deepcopy(self.cfg)

        # resume training
        if cfg.training.resume:
            ckpt_path = pathlib.Path(cfg.checkpoint_dir)
            if ckpt_path.is_file():
                print(f"Resuming from checkpoint {ckpt_path}")
                if cfg.exclude_keys is None:
                    self.load_checkpoint(path=ckpt_path)
                else:
                    self.load_checkpoint(path=ckpt_path, exclude_keys=cfg.exclude_keys)
            self.optimizer = hydra.utils.instantiate( \
                cfg.optimizer, params=self.model.parameters())
            self.global_step = 0
            self.epoch = 0

        device = torch.device(cfg.training.device_gpu)
        ref_policy = copy.deepcopy(self.model)
        # ref_model.double()
        ref_policy.eval() 
        for param in ref_policy.parameters():
            param.requires_grad = False
        ref_policy.to(device)

        # configure dataset
        dataset: BaseImageDataset
        dataset = hydra.utils.instantiate(cfg.task.origin_dataset)
        # assert isinstance(dataset, BaseImageDataset)
        # train_dataloader = DataLoader(dataset, **cfg.dataloader)
        normalizer = dataset.get_normalizer()
        del dataset

        # configure validation dataset
        #val_dataset = dataset.get_validation_dataset()
        #val_dataloader = DataLoader(val_dataset, **cfg.val_dataloader)

        self.model.set_normalizer(normalizer)
        if cfg.training.use_ema:
            self.ema_model.set_normalizer(normalizer)

        # configure dataset
        dataset_1: BaseImageDataset
        dataset_1 = hydra.utils.instantiate(cfg.task.dataset_1)
        #device = torch.device(cfg.training.device_cpu)
        assert isinstance(dataset_1, BaseImageDataset)

        # configure dataset
        dataset_2: BaseImageDataset
        dataset_2 = hydra.utils.instantiate(cfg.task.dataset_2)
        # expert_normalizer = normal_dataset.get_normalizer()
        assert isinstance(dataset_2, BaseImageDataset)

        pref_dataset: BaseImageDataset
        pref_dataset = hydra.utils.instantiate(cfg.task.pref_dataset, dataset_1=dataset_1, \
                                               dataset_2=dataset_2)
        del dataset_1, dataset_2

        # cut online groups
        votes_1, votes_2 = pref_dataset.pref_replay_buffer.meta['votes'], pref_dataset.pref_replay_buffer.meta['votes_2']

        if cfg.training.map.use_map:    

            pref_dataset.pref_replay_buffer.meta['votes'] = votes_1.reshape(-1, 1)
            pref_dataset.pref_replay_buffer.meta['votes_2'] = votes_2.reshape(-1, 1)
            pref_dataset.pref_replay_buffer.root['meta']['votes'] = votes_1.reshape(-1, 1)
            pref_dataset.pref_replay_buffer.root['meta']['votes_2'] = votes_2.reshape(-1, 1)

            pref_dataset.set_beta_priori(obs_encoder = copy.deepcopy(ref_policy.obs_encoder), normalizer=normalizer)
            pref_dataset.beta_model.fit_data(load_dir = 'data/beta_model/itr_20/beta_model.pth')
            # pref_dataset.beta_model.fit_data(dataset=pref_dataset.construct_pref_data(), num_epochs=20, warm_up_epochs=1, batch_size=2, lr=2.0e-5, save_dir='data/beta_model')
            # with torch.no_grad():
            #     pref_dataset.update_beta_priori(batch_size=2)

        train_dataloader = DataLoader(pref_dataset, **cfg.dataloader)

        # for param_group in self.optimizer.param_groups:
        #     if 'initial_lr' not in param_group:
        #         param_group['initial_lr'] = cfg.optimizer.lr

        # # configure lr scheduler
        # lr_scheduler = get_scheduler(
        #     cfg.training.lr_scheduler,
        #     optimizer=self.optimizer,
        #     num_warmup_steps=cfg.training.lr_warmup_steps,
        #     num_training_steps=(
        #         len(train_dataloader) * cfg.training.num_epochs) \
        #             // cfg.training.gradient_accumulate_every,
        #     # pytorch assumes stepping LRScheduler every epoch
        #     # however huggingface diffusers steps it every batch
        #     last_epoch=self.global_step-1
        # )

        # configure ema
        ema: EMAModel = None
        if cfg.training.use_ema:
            ema = hydra.utils.instantiate(
                cfg.ema,
                model=self.ema_model)

        # configure logging
        wandb_run = wandb.init(
            dir=str(self.output_dir),
            config=OmegaConf.to_container(cfg, resolve=True),
            **cfg.logging
        )
        wandb.config.update(
            {
                "output_dir": self.output_dir,
            }
        )

        # configure checkpoint
        topk_manager = TopKCheckpointManager(
            save_dir=os.path.join(self.output_dir, 'checkpoints'),
            **cfg.checkpoint.topk
        )

        # device transfer
        device = torch.device(cfg.training.device_gpu)
        self.model.to(device)
        if self.ema_model is not None:
            self.ema_model.to(device)
        optimizer_to(self.optimizer, device)

        # save batch for sampling
        train_sampling_batch = None

        if cfg.training.debug:
            cfg.training.num_epochs = 2
            cfg.training.max_train_steps = 3
            cfg.training.max_val_steps = 3
            cfg.training.rollout_every = 1
            cfg.training.checkpoint_every = 1
            cfg.training.val_every = 1
            cfg.training.sample_every = 1

        # training loop
        log_path = os.path.join(self.output_dir, 'logs.json.txt')
        with JsonLogger(log_path) as json_logger:
            for online_epoch_idx in range(cfg.training.online.num_groups):
                print(f"Round {online_epoch_idx + 1} of {cfg.training.online.num_groups} for online training")

                n = votes_1.shape[0]
                exchange_count = int(votes_1.shape[0] * cfg.training.online.reverse_ratio)
                indices = np.random.choice(n, exchange_count, replace=False)

                votes_1_locoal = votes_1.copy()
                votes_2_locoal = votes_2.copy()

                temp = votes_1_locoal[indices].copy()
                votes_1_locoal[indices] = votes_2_locoal[indices]
                votes_2_locoal[indices] = temp

                pref_dataset.pref_replay_buffer.meta['votes'] =  votes_1_locoal
                pref_dataset.pref_replay_buffer.meta['votes_2'] = votes_2_locoal
                pref_dataset.pref_replay_buffer.root['meta']['votes'] =  votes_1_locoal
                pref_dataset.pref_replay_buffer.root['meta']['votes_2'] = votes_2_locoal

                train_dataloader = DataLoader(pref_dataset, **cfg.dataloader)

                self.optimizer = hydra.utils.instantiate( \
                    cfg.optimizer, params=self.model.parameters())

                for param_group in self.optimizer.param_groups:
                    if 'initial_lr' not in param_group:
                        param_group['initial_lr'] = cfg.optimizer.lr

                lr_scheduler = get_scheduler(
                    cfg.training.lr_scheduler,
                    optimizer=self.optimizer,
                    num_warmup_steps=cfg.training.lr_warmup_steps,
                    num_training_steps=(
                        len(train_dataloader) * cfg.training.num_epochs) \
                            // cfg.training.gradient_accumulate_every,
                    last_epoch=-1,
                )

                for local_epoch_idx in range(cfg.training.num_epochs):
                    step_log = dict()
                    # ========= train for this epoch ==========
                    if cfg.training.freeze_encoder:
                        self.model.obs_encoder.eval()
                        self.model.obs_encoder.requires_grad_(False)

                    train_losses = list()
                    with tqdm.tqdm(train_dataloader, desc=f"Training epoch {self.epoch}", 
                            leave=False, mininterval=cfg.training.tqdm_interval_sec) as tepoch:
                        for batch_idx, batch in enumerate(tepoch):
                            # device transfer
                            batch = dict_apply(batch, lambda x: x.to(device, non_blocking=True))
                            if train_sampling_batch is None:
                                train_sampling_batch = batch

                            # compute loss
                            avg_traj_loss = 0.0
                            if cfg.training.map.use_map:
                                avg_traj_loss = compute_all_traj_loss_realrobot(replay_buffer = pref_dataset.pref_replay_buffer, \
                                                                      model = self.model, ref_model = ref_policy.model, stride=cfg.stride) #change
                            raw_loss = self.model.compute_loss(batch, stride=cfg.stride, ref_model=ref_policy, avg_traj_loss=avg_traj_loss)
                            loss = raw_loss / cfg.training.gradient_accumulate_every
                            loss.backward()

                            # step optimizer
                            if self.global_step % cfg.training.gradient_accumulate_every == 0:
                                self.optimizer.step()
                                self.optimizer.zero_grad()
                                lr_scheduler.step()
                            
                            # update ema
                            if cfg.training.use_ema:
                                ema.step(self.model)

                            # logging
                            raw_loss_cpu = raw_loss.item()
                            tepoch.set_postfix(loss=raw_loss_cpu, refresh=False)
                            train_losses.append(raw_loss_cpu)
                            step_log = {
                                'train_loss': raw_loss_cpu,
                                'global_step': self.global_step,
                                'epoch': self.epoch,
                                'lr': lr_scheduler.get_last_lr()[0]
                            }

                            is_last_batch = (batch_idx == (len(train_dataloader)-1))
                            if not is_last_batch:
                                # log of last step is combined with validation and rollout
                                wandb_run.log(step_log, step=self.global_step)
                                json_logger.log(step_log)
                                self.global_step += 1

                            if (cfg.training.max_train_steps is not None) \
                                and batch_idx >= (cfg.training.max_train_steps-1):
                                break

                    # at the end of each epoch
                    # replace train_loss with epoch average
                    train_loss = np.mean(train_losses)
                    step_log['train_loss'] = train_loss

                    # ========= eval for this epoch ==========
                    policy = self.model
                    if cfg.training.use_ema:
                        policy = self.ema_model
                    policy.eval()

                    if (self.epoch % cfg.training.sample_every) == 0:
                        with torch.no_grad():
                            # sample trajectory from training set, and evaluate difference
                            batch = dict_apply(train_sampling_batch, lambda x: x.to(device, non_blocking=True))
                            obs_dict = batch['obs']
                            obs_dict_2 = batch['obs_2']
                            for key in obs_dict.keys():
                                obs_dict[key] = obs_dict[key][:, :self.model.n_obs_steps, ...]

                            for key in obs_dict_2.keys():
                                obs_dict_2[key] = obs_dict_2[key][:, :self.model.n_obs_steps, ...]

                            gt_action = batch['action'][:, self.model.n_obs_steps:self.model.n_obs_steps+self.model.n_action_steps, ...]
                            gt_action_2 = batch['action_2'][:, self.model.n_obs_steps:self.model.n_obs_steps+self.model.n_action_steps, ...]
                            
                            result = policy.predict_action(obs_dict)
                            pred_action = result['action']

                            result_2 = policy.predict_action(obs_dict_2)
                            pred_action_2 = result_2['action']

                            mse = torch.nn.functional.mse_loss(pred_action, gt_action)
                            mse_2 = torch.nn.functional.mse_loss(pred_action_2, gt_action_2)
                            
                            step_log['train_action_mse_error'] = (mse.item() + mse_2.item()) * 0.5

                            del batch, obs_dict, gt_action, result, pred_action, mse, obs_dict_2, gt_action_2, result_2, pred_action_2, mse_2

                    # checkpoint
                    if (self.epoch % cfg.training.checkpoint_every) == 0:
                        # checkpointing
                        if cfg.checkpoint.save_last_ckpt:
                            self.save_checkpoint()
                        if cfg.checkpoint.save_last_snapshot:
                            self.save_snapshot()

                        # sanitize metric names
                        metric_dict = dict()
                        for key, value in step_log.items():
                            new_key = key.replace('/', '_')
                            metric_dict[new_key] = value
                        
                        # We can't copy the last checkpoint here
                        # since save_checkpoint uses threads.
                        # therefore at this point the file might have been empty!
                        topk_ckpt_path = topk_manager.get_ckpt_path(metric_dict)

                        if topk_ckpt_path is not None:
                            self.save_checkpoint(path=topk_ckpt_path)
                    # ========= eval end for this epoch ==========
                    policy.train()

                    # end of epoch
                    # log of last step is combined with validation and rollout
                    wandb_run.log(step_log, step=self.global_step)
                    json_logger.log(step_log)
                    self.global_step += 1
                    self.epoch += 1

@hydra.main(
    version_base=None,
    config_path=str(pathlib.Path(__file__).parent.parent.joinpath("config")), 
    config_name=pathlib.Path(__file__).stem)
def main(cfg):
    workspace = PbrlDiffusionRealRobotWorkspace(cfg)
    workspace.run()

if __name__ == "__main__":
    main()
